# Simple partially pivoted ACA

# Returns (U, V) factorization of f(X, Y)
# X, Y are d_x x m and d_y x n respectively
function pp_aca(f, X::Array{Float64, 2}, Y::Array{Float64, 2}, tol::Float64 ; fixedRank=nothing, loglevel::SI.LogLevel=SI.none, Ktrue=nothing)
    m = size(X)[2]
    n = size(Y)[2]
    U = Array{Array{Float64,1},1}(0)
    V = Array{Array{Float64,1},1}(0)
    Z = Array{Int64, 1}(0)
    I = Array{Int64, 1}(0)
    J = Array{Int64, 1}(0)
    # Find first basis
    i1 = 1
    push!(Z, i1)
    v1 = f(X[:,i1], Y)
    assert(norm(v1) > 0)
    (trash,j1) = findmax(abs.(v1))
    v1 = v1/v1[j1]
    u1 = f(X, Y[:,j1]) # K_{i, j1} for all i
    normS = norm(u1)^2*norm(v1)^2
    push!(U, u1)
    push!(V, v1)
    push!(I, i1)
    push!(J, j1)
    for k = 1:min(m,n)-1
        notZ = sort(setdiff(1:m, Z))
        (trash,ik) = findmax(abs.(U[end][notZ]))
        ik = notZ[ik]
        push!(Z, ik)
        vk = f(X[:,ik], Y)
        for l = 1:k
            vk = vk - U[l][ik] * V[l]
        end
        (trash,jk) = findmax(abs.(vk))
        vk = vk / vk[jk]
        uk = f(X, Y[:,jk])
        for l = 1:k
            uk = uk - U[l] * V[l][jk]
        end
        push!(U,uk)
        push!(V,vk)
        push!(I, ik)
        push!(J, jk)
        for l = 1:k
            normS = normS + 2 * (uk'*U[l])[1] * (V[l]'*vk)[1]
        end
        err = norm(vk)*norm(uk)
        normS = normS + err^2
        if loglevel >= SI.debug
            @printf "ACA - selecting (%d, %d)\n" ik jk
            @printf "ACA %d - %e <?< %e sqrt(%e) = %e\n" k err tol normS tol*sqrt(normS)
        end
        if fixedRank != nothing
            if k == fixedRank-1
                if loglevel >= SI.verbose
                    @printf "ACA - rank criterion reached.\n"
                end
                break
            end
        elseif Ktrue != nothing
            Utmp = hcat(U...)
            Vtmp = hcat(V...)
            err = vecnorm(Ktrue - Utmp*Vtmp')/vecnorm(Ktrue)
            if err < tol
                if loglevel >= SI.verbose
                    @printf "ACA - true error criterion reached\n"
                end
                break
            end
        else
            if err < tol * sqrt(normS)
                if loglevel >= SI.verbose
                    @printf "ACA - converged.\n"
                end
                break
            end
        end
    end
    U = hcat(U...)
    V = hcat(V...)
    return U, V, I, J
end
