# Build all test geometries
using Iterators

function getGeometriesNames(coef=0)
    return ["Parallel plates distance 1","Parallel plates distance 3", "Perpendicular plates","45 degrees plates","Facing ellipsoids 2d","Facing ellipsoids 3d","Torus Angle 2","Torus Angle 8","Torus Angle 16",string("Squares distance ", coef+1.0),string("Cubes distance ", coef+1.0),"Ellipsoids 2d Angle","Parallel plates non-uniform distribution"]
end

# Geometries :
# 1. & 2. Parallel Plates at distance 1 or 3. Plates are 1 x 1.5, dim_x = 3, dim_r = 2
# 3. Perpendicular Plates
# 4. 45 Degrees Plates
# 5. Facing Ellipsoids 2d
# 6. Facing Ellipsoids 3d
# 7. & 8. & 9. Parts of Torus
# 10. 2d Squares
# 11. 3d Cubes
# 12. Rotated 2d Ellipsoid
# 13. Parallel Plates at distance 3 with non-uniform distribution 
function getGeometry(geoId::Int64, npts::Int64; coef=0)

    @assert 1 <= geoId && geoId <= 13
    @assert npts >= 0
    @printf "Generating geometry %d (%s)\n" geoId getGeometriesNames()[geoId]

    rrt = Array{Array{Float64, 1}, 1}(3)
    for d = 1:3
        rrt[d] = linspace(0, 1, npts)
    end
    RRt2d = SI.tensor_grid(rrt[1:2])

    if geoId == 1 || geoId == 2 # Parallel plates
        if geoId == 1
            X = broadcast(+, [-0.5, 0, 0], [0 0 ; 1 0 ; 0 1] * [1 0 ; 0 1.5] * RRt2d)
            Y = broadcast(+, [ 0.5, 0, 0], [0 0 ; 1 0 ; 0 1] * [1 0 ; 0 1.5] * RRt2d)
        else
            X = broadcast(+, [-1.5, 0, 0], [0 0 ; 1 0 ; 0 1] * [1 0 ; 0 1.5] * RRt2d)
            Y = broadcast(+, [ 1.5, 0, 0], [0 0 ; 1 0 ; 0 1] * [1 0 ; 0 1.5] * RRt2d)
        end
        b1 = SI.CubeBoundingBox{Float64}(X, compress=true)
        b2 = SI.CubeBoundingBox{Float64}(Y, compress=true)
    elseif geoId == 3 # Perpendicular plates
        X = broadcast(+, [-1.0, 0, 0],  [0 0 ; 1 0 ; 0 1] * [1 0 ; 0 1.5] * RRt2d)
        Y = broadcast(+, [ 0, 0, -1.0], [1 0 ; 0 1 ; 0 0] * [1.5 0 ; 0 1] * RRt2d)
        b1 = SI.CubeBoundingBox{Float64}(X, compress=true)
        b2 = SI.CubeBoundingBox{Float64}(Y, compress=true)
    elseif geoId == 4 # 45 degrees plates
        X = broadcast(+, [-0.5, 0, 0], [0 0 ; 1 0 ; 0 1] * [1 0 ; 0 1.5] * RRt2d)
        Y = broadcast(+, [ 0.5, 0, 0], [0 1/sqrt(2) ; 1 0 ; 0 1/sqrt(2)] * [1 0 ; 0 1.5] * RRt2d)
        b1 = SI.CubeBoundingBox{Float64}(X, compress=true)
        b2 = SI.CubeBoundingBox{Float64}(Y, compress=true)
    elseif geoId == 5 || geoId == 6 || geoId == 12 # Ellipsoid 2d and 3d, rotated for 12
        fileVertices  = "meshes/ellipsoid-0.2.vtx"
        fileTriangles = "meshes/ellipsoid-0.2.tgl"
        mesh = SI.TriangularMesh(fileVertices, fileTriangles) 
        # Create two partitions
        p1_t = Array{Int64, 1}(0)
        p2_t = Array{Int64, 1}(0)
        if geoId == 5 || geoId == 6
            for i = 1:size(mesh.centers)[2]
                if mesh.centers[3,i] > 0.7
                    push!(p1_t, i)
                elseif mesh.centers[3,i] < -0.7
                    push!(p2_t, i)
                end
            end
        elseif geoId == 12
            for i = 1:size(mesh.centers)[2]
                if mesh.centers[3,i] > 0.1 && mesh.centers[1,i] < -0.5
                    push!(p1_t, i)
                elseif mesh.centers[3,i] < -0.1 && mesh.centers[1,i] > 0.5
                    push!(p2_t, i)
                end
            end
        end
        p1_v = mesh.triangles[:,p1_t][:]
        p2_v = mesh.triangles[:,p2_t][:]
        # Collect vertices
        mesh.centers[3,p1_t] = mesh.centers[3,p1_t]+1
        mesh.centers[3,p2_t] = mesh.centers[3,p2_t]-1
        mesh.vertices[3,p1_v] = mesh.vertices[3,p1_v]+1
        mesh.vertices[3,p2_v] = mesh.vertices[3,p2_v]-1
        # Extract required stuff for bounding box
        V1 = mesh.vertices[:,p1_v]
        V2 = mesh.vertices[:,p2_v]
        C1 = mesh.centers[:,p1_t]
        C2 = mesh.centers[:,p2_t]
        # Extract required stuff for bounding box
        if geoId == 6 # 3d
            X = C1
            Y = C2 
            b1 = SI.EllipsoidBoundingBox{Float64}(hcat(V1,C1))
            b2 = SI.EllipsoidBoundingBox{Float64}(hcat(V2,C2))
        else geoId == 5 || geoId == 12 # 2d
            X = V1
            Y = V2
            b1 = SI.EllipsoidBoundingBox{Float64}(X, compress=true, tol=1e-4)
            b2 = SI.EllipsoidBoundingBox{Float64}(Y, compress=true, tol=1e-4)
            X = SI.RtoX(b1,SI.XtoR(b1, X))
            Y = SI.RtoX(b2,SI.XtoR(b2, Y))
            b1 = SI.EllipsoidBoundingBox{Float64}(X, compress=true)
            b2 = SI.EllipsoidBoundingBox{Float64}(Y, compress=true)
        end
    elseif geoId == 7 || geoId == 8 || geoId == 9
        deltaPhi = 2*pi/32.0
        if geoId == 7
            idx = 2
        elseif geoId == 8
            idx = 8
        elseif geoId == 9
            idx = 16
        end
        # @printf "Torus with coef %d\n" idx
        b1 = SI.TorusBoundingBox{Float64}(10.0, 1.0, [0.01,pi-0.01], [-pi+0.01, -pi+0.01+deltaPhi])
        b2 = SI.TorusBoundingBox{Float64}(10.0, 1.0, [0.01,pi-0.01], [-pi+0.01+idx*deltaPhi,-pi+0.01+(idx+1)*deltaPhi])
        X = SI.generatePoints(b1,npts,npts)
        Y = SI.generatePoints(b2,npts,npts)
    elseif geoId == 10
        b1 = SI.CubeBoundingBox{Float64}([0.0,0.0],[1.0,1.0])
        b2 = SI.CubeBoundingBox{Float64}([2.0+coef,2.0+coef],[1.0,1.0])
        # @printf "Generating boxes with coef %e\n" coef
        X = SI.generatePoints(b1, npts * [1,1])
        Y = SI.generatePoints(b2, npts * [1,1])
    elseif geoId == 11
        b1 = SI.CubeBoundingBox{Float64}([0.0,0.0,0.0],[1.0,1.0,1.0])
        b2 = SI.CubeBoundingBox{Float64}([2.0+coef,2.0+coef,2.0+coef],[1.0,1.0,1.0])
        X = SI.generatePoints(b1, npts * [1,1,1])
        Y = SI.generatePoints(b2, npts * [1,1,1])
    elseif geoId == 13
        RNG = srand(0)
        # Centered at (0.5, 0.5) and (1.5, 1.2)
        X = 0.2 * randn(RNG, (2, npts * npts)) + 0.5
        Y = 0.2 * randn(RNG, (2, npts * npts)) + 2.5
        Xid = (X[1,:] .> 0.0) & (X[1,:] .< 1.0) & (X[2,:] .> 0.0) & (X[2,:] .< 1.0) 
        Yid = (Y[1,:] .> 2.0) & (Y[1,:] .< 3.0) & (Y[2,:] .> 2.0) & (Y[2,:] .< 3.0) 
        X = X[:,Xid]
        Y = Y[:,Yid]
        # Add some extreme points
        # Xext = broadcast(+, rand(RNG, (2, 2)), [1.0, 0.0])
        # Yext = broadcast(+, rand(RNG, (2, 2)), [1.0, 2.0])
        Xext = [2.0 2.0 -1.0 -1.0 ;
                0.0 1.0 0.0 1.0];
        Yext = [1.0 1.0 4.0 4.0 ;
                2.0 3.0 2.0 3.0];
        X = hcat(X, Xext) 
        Y = hcat(Y, Yext)
        X[2,:] -= 1
        Y[2,:] += 1
        b1 = SI.CubeBoundingBox{Float64}([-1.0,-1.0],[3.0,1.0]) # npts x npts points, not uniformly distributed
        b2 = SI.CubeBoundingBox{Float64}([1.0,3.0],[3.0,1.0])
    else
        @assert false
    end
    return (X, Y, b1, b2) 
end
