include("../src/SI.jl")
using Interpolations
using Polynomials
using PlotlyJS
PyPlot.close("all")
srand(0)

kernel = (x,y) -> 1./(4+broadcast(-, x[1,:], y[1,:]))

n = 1000
b1 = SI.CubeBoundingBox{Float64}([-1.0],[2.0])
b2 = SI.CubeBoundingBox{Float64}([-1.0],[2.0])
X = reshape(Array(linspace(-1,1,n+1)[1:end-1]), (1, n))
Y = reshape(Array(linspace(-1,1,n+1)[1:end-1]), (1, n))
dS = 1.0/n
tol = 1e-5
tolQR = 1e-6
Ktrue = SI.meshKernelFull(kernel, X, Y)

# We monitor using MDV points in X and Y
Xbar = n -> (X[:,SI.get_mdv(X,n)], ones(n))
Ybar = n -> (Y[:,SI.get_mdv(Y,n)], ones(n))
stop = (r0, r1, Xhat, Yhat) -> SI.accuracy_stopping_criterion(kernel, X, Y, Xhat, Yhat, Ktrue, tol)
(Xhat, Yhat, Xbar, Ybar) = SI.adaptive_si(kernel, Xbar, Ybar, tolQR, rinit=5, rmax=1000, rinc=x->ceil(x*1.1), loglevel=SI.debug, stopping_criterion=stop)
r0 = size(Xbar)[2]
r1 = size(Xhat)[2]
@printf "r0 %d, r1 %d\n" r0 r1

# SI
Khat      = SI.meshKernelFull(kernel, Xhat, Yhat)
Kx        = SI.meshKernelFull(kernel, X, Yhat)
Ky        = SI.meshKernelFull(kernel, Xhat, Y)
@printf "Error %e\n" vecnorm(Kx * (Khat \ Ky) - Ktrue)/vecnorm(Ktrue)

# True eigenfunctions
(U,s,V)  = svd(Ktrue)
U        = U / sqrt(dS)
V        = V / sqrt(dS)
s        = s * dS
uTrue = []
vTrue = []
for i = 1:r1+1
    uTruei = interpolate(U[:,i], BSpline(Cubic(Line())), OnGrid())
    uTrueiFunc = scale(uTruei, linspace(minimum(X[1,:]), maximum(X[1,:]), n))
    uTrueiFuncx = x -> uTrueiFunc[x]
    push!(uTrue, uTrueiFuncx)
    vTruei = interpolate(V[:,i], BSpline(Cubic(Line())), OnGrid())
    vTrueiFunc = scale(vTruei, linspace(minimum(Y[1,:]), maximum(Y[1,:]), n))
    vTrueiFuncy = y -> vTrueiFunc[y]
    push!(vTrue, vTrueiFuncy)
end
@printf "Done building ui(x) and vi(y)\n"

# Check orthogonality of r1 eigenfunctions over **Xbar** and **Ybar** which have r0 points
uTrueXbar = zeros(r0, r1)
vTrueYbar = zeros(r0, r1)
for j = 1:r1
    uTrueXbar[:,j] = uTrue[j].(Xbar[1,:])
    vTrueYbar[:,j] = vTrue[j].(Ybar[1,:])
end
uTrueXhat = zeros(r1, r1)
vTrueYhat = zeros(r1, r1)
for j = 1:r1
    uTrueXhat[:,j] = uTrue[j].(Xhat[1,:])
    vTrueYhat[:,j] = vTrue[j].(Yhat[1,:])
end
uTrueX = zeros(n, r1+1)
vTrueY = zeros(n, r1+1)
for j = 1:r1+1
    uTrueX[:,j] = uTrue[j].(X[1,:])
    vTrueY[:,j] = vTrue[j].(Y[1,:])
end

# How does U_X^T R behaves ?
SXXhat = Kx / Khat
R = SXXhat * uTrueXhat - uTrueX[:,1:r1]
factor =  1.0 / ( 1.0 - norm(1.0/n * uTrueX' * R) )
@show value = norm(inv(uTrueXhat))
@show num = norm(SXXhat)/sqrt(n) # The right value !
@show bound = num * factor

# Plot
ts = Array{PlotlyJS.GenericTrace, 1}()
for i = 1:r1
    push!(ts, scatter(x=X[1,:], y=uTrueX[:,i], mode="line", name=string("u", i, "(x)")))
end
#push!(ts, scatter(x=X[1,:], y=uTrueX[:,r1+1], mode="line", name="ur+1(x)"))
push!(ts, scatter(x=Xhat[1,:], y=zeros(r1), mode="marker", size=10))
plt1 = plot(ts)
display(plt1)

ts = Array{PlotlyJS.GenericTrace, 1}()
for i = 1:r1
    push!(ts, scatter(x=X[1,:], y=abs.(R[:,i]), mode="line", name=string("u", i, "(x) error")))
end
lyt = Layout(yaxis_type="log", yaxis_exponentformat="e");
plt2 = plot(ts, lyt);
display(plt2);

