# Compare mappings
include("../src/SI.jl")
include("../benchmarks/geometries.jl")
using PyPlot
using LowRankApprox
using JLD
using Dates
close("all")

# Define problem
rule = SI.chebyshev
function dist(x, y)
    X_Y = broadcast(-, x, y)
    return sqrt(sum(X_Y.^2, 1))[:]
end
kernel = (x, y) -> 1.0./dist(x, y)
npts = 30
nrepeat = 50
(X, Y, b1, b2) = getGeometry(13, npts)
Xu = SI.generatePoints(b1, 10 * [1,1])
Yu = SI.generatePoints(b2, 10 * [1,1])

figure(1)
SI.plotBoundingBox(b1, 1);
SI.plotBoundingBox(b2, 1);
plot(X[1,:],X[2,:],"ok",markersize=2)
plot(Y[1,:],Y[2,:],"ok",markersize=2)

# Run Algorithm
tols = logspace(-1, -8, 8)
Atrue = SI.meshKernelFull(kernel, X, Y)
r_opts = SI.rank_eps_l2(Atrue, tols)
r_opts_l2 = SI.rank_eps_l2(SI.meshKernelFull(kernel, Xu, Yu), tols)
r_si = zeros(Int64, length(tols))
r_rand = zeros(Int64, (length(tols), nrepeat))
err_si = zeros(Float64, length(tols))
err_rand = zeros(Float64, (length(tols), nrepeat))
err_si_inf = zeros(Float64, length(tols))
err_rand_inf = zeros(Float64, (length(tols), nrepeat))

# Reduced A
Ared = SI.meshKernelFull(kernel, X[:,1:end-4], Y[:,1:end-4])
AredY = SI.meshKernelFull(kernel, X, Y[:,1:end-4])
AredX = SI.meshKernelFull(kernel, X[:,1:end-4], Y)

# Scope issues
px = 0
py = 0
rows = 0
cols = 0
Xk = 0
Yk = 0
Iaca = 0
Jaca = 0

opts = LRAOptions()
#opts.sketch = :sub

for (t, tol) in enumerate(tols) 
    ## SI
    (U, F, V, Xk, Yk, Uuc, Fuc, Vuc, I1, I2, diags) = SI.meshKernelLowRankAdaptive(kernel, b1, b2, X, Y, tol, ruletype=rule, useSI=true) 
    r_si[t] = minimum(size(F))
    Aerr = U*(F\V') - Atrue
    err_si[t] = norm(Aerr)/norm(Atrue)
    err_si_inf[t] = maximum(abs(Aerr)[:])/maximum(abs(Atrue)[:])
    ## Random RRLU
    for i = 1:nrepeat
        px = randperm(size(X)[2])[1:r_si[t]]
        py = randperm(size(Y)[2])[1:r_si[t]]
        Xr = X[:,px]
        Yr = Y[:,py]
        Aerr = SI.meshKernelFull(kernel, X, Yr) * (SI.meshKernelFull(kernel, Xr, Yr) \ SI.meshKernelFull(kernel, Xr, Y)) - Atrue
        err_rand[t,i] = norm(Aerr) / norm(Atrue)
        err_rand_inf[t,i] = maximum(abs(Aerr[:]))/maximum(abs(Atrue[:]))
    end
end
plot(Xk[1,:],Xk[2,:],"^r",markersize=3,label="SI")
plot(Yk[1,:],Yk[2,:],"^r",markersize=3)
legend()

figure(2)
loglog(tols, err_si_inf,label="SI")
loglog(tols, mean(err_rand_inf, 2),label="Mean Random RRLU")
loglog(tols, tols,label="Tol")
xlabel("Tol")
ylabel("Error (Inf norm)")
legend()

figure(3)
semilogx(tols, r_si)
semilogx(tols, r_opts)
semilogx(tols, r_opts_l2)
legend(("SI","SVD","Cont. SVD"))
xlabel("Tol")
ylabel("Rank")
