#
# Does the generic SI adaptive algorithm
# Accuracy is based on the CUR ranks
#

function accuracy_stopping_criterion(kernel, X, Y, Xhat, Yhat, Ktrue, tol, loglevel=SI.verbose)
    Kapp = meshKernelFull(kernel, X, Yhat) * (meshKernelFull(kernel, Xhat, Yhat) \ meshKernelFull(kernel, Xhat, Y)) 
    err = vecnorm(Kapp - Ktrue) / vecnorm(Ktrue)
    if loglevel >= info
        @printf "Stopping ? r1 %d, err %e\n" size(Xhat, 2) err
    end
    return (err <= tol, (size(Xhat, 2), err))
end

# Return wether we should stop (true) or continue (false)
function default_stopping_criterion(r0, r1, Xhat, Yhat)
    return (r1 < r0 * 0.75 && r1 < r0 - 10), nothing
end

# The function finds Xhat, Yhat such that the CUR
# of K(Xmon, Ybar) and K(Xbar, Ymon)
# give rank r1 much less than the sizes of min(|Xmon|,|Ybar|)
# and min(|Xbar|,|Ymon|) (r0)

# Xhat and Yhat are extracted from Xbar and Ybar
# Xhat and Yhat are selected by doing:
# RRQR on K(Xmon, Ybar)   --> finds Yhat
# RRQR on K(Xbar, Ymon)^T --> finds Xhat
# Potentially, Xmon = Xbar and Ymon = Ybar (like Cheb)
# Weights can also be added if desired. If so, 
# RRQR's are done on Wx^1/2 K(X, Y) Wy^1/2
# instead

# Requirements:
#       For any r0, should be able to generate
#       Xbar, Ybar, Xmon, Ymon
#       of a size related to r0 (in practice, they can have a fixed size)

# Arrays are always [dim x N]
# where dim is the space dimension (typically, 2, 3)
# and N is the # of points in the set

# kernel
#       function (x, y) --> k(x, y), x and y are [dim x N], k(x, y) is [N]
# Xmon, Ymon, Xbar, Ybar
#       functions (n) --> (X,W) = ([dim x N],[N]) 
#       outputs the points and the weights
# rinit, rmax
#       initial and maximal ranks
# rinc 
#       function (r0) --> (r1)
#       gives the next rank
# tols   
#       the tolerances for the RRQR's
#       will try multiple and stops at the first one that works
# loglevel
#       the log level
function adaptive_si(kernel, Xbar, Ybar, tols ; Xmon=nothing, Ymon=nothing, rinit=10, rmax=100, rinc=x->x*2, loglevel=SI.verbose, stopping_criterion=default_stopping_criterion)
   
    @show tols
    px = nothing
    py = nothing
    XbarA = nothing
    YbarA = nothing
    diags = []
    r0s   = []
    while true
        r0 = rinit
        r1 = nothing
        # Get the sets
        (XbarA, Wxbar) = Xbar(r0)
        (YbarA, Wybar) = Ybar(r0)
        if Xmon == nothing
            (XmonA, Wxmon) = (XbarA, Wxbar)
        else
            (XmonA, Wxmon) = Xmon(r0)
        end
        if Ymon == nothing
            (YmonA, Wymon) = (YbarA, Wybar)
        else
            (YmonA, Wymon) = Ymon(r0)
        end
        # Build K(Xmon, Ybar)
        K0y = diagm(sqrt.(Wxmon)) * meshKernelFull(kernel, XmonA, YbarA) * diagm(sqrt.(Wybar))
        # Build K(Xbar, Ymon) (transpose of the previous one usually)
        K0x = diagm(sqrt.(Wxbar)) * meshKernelFull(kernel, XbarA, YmonA) * diagm(sqrt.(Wymon))
        # Find Xhat, Yhat
        for tol in tols
            (px, py) = cur2(K0x, K0y, tol)
            @assert length(px) == length(py)
            r0x = minimum(size(K0x))
            r0y = minimum(size(K0y))
            r1 = length(px)
            if loglevel >= debug
                @printf "Ranks r0 %d, r1 %d, tol %e (Kx is %dx%d, Ky is %dx%d)\n" r0 r1 tol size(K0x,1) size(K0x,2) size(K0y,1) size(K0y,2)
            end
            # Converged ?
            (stop, diag) = stopping_criterion(r0, r1, XbarA[:,px], YbarA[:,py])
            push!(diags, diag)
            push!(r0s, r0)
            if stop
                if loglevel >= info
                    @printf "Converged with r0 %d, r1 %d\n" r0 r1
                end
                return (XbarA[:,px], YbarA[:,py], XbarA, YbarA, diags, r0s)
            end
        end
        rinit = rinc(r0)
        if rinit > rmax
            if loglevel >= info
                @printf "Aborted with r0 %d, r1 %d\n" r0 r1
            end
            break
        end
    end
    return (XbarA[:,px], YbarA[:,py], XbarA, YbarA, diags, r0s)
end

    

