type TriangularMesh{T}

    vertices::Array{T,2}      # dim x V
    centers::Array{T,2}       # dim x T
    triangles::Array{Int64,2}       # 3 x T

    # Files are
    # vertexfile [numVertices x dimension]
    # trianglefile [numTriangles x 3]
    function TriangularMesh(vertexfile, trianglefile)
        this = new()
        this.vertices = readdlm(vertexfile, T)'
        dim = size(this.vertices)[1]
        this.triangles = readdlm(trianglefile, Int64)' 

        # Check size and consistency
        @assert size(this.triangles)[1] == 3
        sortedAndUnique = sort(unique(this.triangles[:]))
        maxSortedAndUnique = maximum(sortedAndUnique)
        @assert sortedAndUnique == collect(1:maxSortedAndUnique) 
        @assert size(this.vertices)[2] == maxSortedAndUnique

        # Number of vertices and triangles
        nT = size(this.triangles)[2]
        nV = size(this.vertices)[2] 
        
        # Compute the centers
        this.centers = Array{T,2}(dim, nT)
        for t = 1:nT
            for d = 1:dim
                this.centers[d,t] = mean(this.vertices[d,this.triangles[:,t]])
            end
        end

        # Return this
        this
    end

end

function plot(mesh::TriangularMesh, figure)
   
    PyPlot.figure(figure)

    for t = 1:size(mesh.triangles)[2]
        
        t = mesh.triangles[:,t]
        V = mesh.vertices[:,t]
        PyPlot.plot3D(V[1,[1 2]][:],V[2, [1 2]][:],V[3, [1 2]][:],"-k")
        PyPlot.plot3D(V[1,[1 3]][:],V[2, [1 3]][:],V[3, [1 3]][:],"-k")
        PyPlot.plot3D(V[1,[2 3]][:],V[2, [2 3]][:],V[3, [2 3]][:],"-k")
    
    end
    
end
