@enum RULETYPE chebyshev=1 legendre=2 uniform=3 norule=4 # The type of 1d rule

using FastGaussQuadrature

# Return (n, w, x, w_int)
# # of points == order
# w_int gives the weight of nodes in any integration rule
function getChebyshevNodes(order, a, b)
    @assert a <= b
    n = order
    k = n:-1:1
    x = (a+b)/2.0 + (b-a)/2.0*cos.((2*k-1)/(2*n)*pi)
    w = barycentric_weights(x)
    w_int = (b - a) / 2 * pi / n * sin.((2*k-1)/(2*n)*pi)
    @assert all(w_int .> 0)
    return (n, w, x, w_int)
end

function getLegendreNodes(order, a, b)
    @assert a <= b
    n = order
    (nodes, weights) = gausslegendre(n) # Order is irrelevant
    x = (a+b)/2.0 + (b-a)/2.0*nodes
    w = barycentric_weights(x)
    w_int = (b - a) / 2 .* weights
    @assert all(w_int .> 0)
    return (n, w, x, w_int)
end

function getUniformNodes(order, a, b)
    @assert a <= b
    if order == 1
        n = 1
        x = [(b-a)/2]
        w_int = [(b-a)]
        w = barycentric_weights(x)
    else
        n = order
        x = Array{Float64, 1}(linspace(a, b, n))
        w = barycentric_weights(x)
        w_int = ones(size(x)) * (b - a) / (n - 1)
        w_int[1] *= 0.5
        w_int[end] *= 0.5
    end
    return (n, w, x, w_int)
end
