
@enum LogLevel none=0 info=1 verbose=2 debug=3

function print_log(str, logLevel::LogLevel, logLevelRequired::LogLevel)
    if logLevel >= logLevelRequired
        println(str)
    end
end

function distance_2{T}(x::Array{T, 1}, y::Array{T, 1})
    @assert size(x) == size(y)
    return sqrt.(sum((x - y).^2,1))
end

function distance_2{T}(x::Array{T, 2}, y::Array{T, 2})
    @assert size(x) == size(y)
    return sqrt.(sum((x - y).^2,1))[:]
end

function distance_2{T}(x::Array{T, 1}, y::Array{T, 2})
    return sqrt.(sum((broadcast(-, x, y)).^2,1))[:]
end

function distance_2{T}(x::Array{T, 2}, y::Array{T, 1})
    return sqrt.(sum((broadcast(-, x, y)).^2,1))[:]
end

# Given a list of sorted values [s1 ... sn]
# returns an index i such that
# || si+1 ... send ||_2 ~= tol * || s1 ... sn ||_2
# by finding the first i such that
# || si+1 ... send ||_2 <= tol * || s1 ... sn ||_2
# Can return any integer from 0 (for tol >= 1) to n
function cut_spectrum_fro(s, tol; bypasssort=false)
    if ! bypasssort
        @assert issorted(s, rev=true)
    else
        if ! issorted(s, rev=true)
            warn(string("Not sorted array : ", string(s)))
        end
    end
    @assert all(s .> 0)
    n  = length(s)
    ns = vecnorm(s)
    for j = 0:n-1
        if vecnorm(s[j+1:end]) <= tol * ns
            return j
        end
    end
    return n
end

function cut_spectrum_l2(s, tol; bypasssort=false)
    if ! bypasssort
        @assert issorted(s, rev=true)
    else
        if ! issorted(s, rev=true)
            warn(string("Not sorted array : ", string(s)))
        end
    end
    @assert all(s .>= 0)
    n  = length(s)
    ns = s[1]
    for j = 0:n-1
        if s[j+1] <= tol * ns
            return j
        end
    end
    return n
end

function rank_eps_fro(A, tols ; ll::LogLevel=SI.info)
    (U,s,V) = svd(A)
    @assert issorted(s, rev=true)
    ranks = Array{Int64, 1}(length(tols))
    for i = 1:length(tols)
        j = cut_spectrum_fro(s, tols[i])
        ranks[i] = j
        # Actually check accuracy
        err = vecnorm(U[:,1:j]*diagm(s[1:j])*V[:,1:j]' - A)/vecnorm(A)
        if ll >= SI.verbose
            @printf "%d. Rank %d gives error %e (required %e)\n" i ranks[i] err tols[i]
        end
    end
    return ranks
end

function rank_eps_l2(A, tols ; ll::LogLevel=SI.info)
    (U,s,V) = svd(A)
    @assert issorted(s, rev=true)
    ranks = Array{Int64, 1}(length(tols))
    for i = 1:length(tols)
        j = cut_spectrum_l2(s, tols[i])
        ranks[i] = j
        # Actually check accuracy
        err = norm(U[:,1:j]*diagm(s[1:j])*V[:,1:j]' - A, 2)/norm(A, 2)
        if ll >= SI.verbose
            @printf "%d. Rank %d gives error %e (required %e)\n" i ranks[i] err tols[i]
        end
    end
    return ranks
end

function rank_eps(A, tol)
    (U,S,V) = svd(A) 
    idxToKeep = find(x -> x >= S[1]*tol, S)
    return length(idxToKeep)
end

function errorAppTrue(fapp, ftrue)
    @assert maximum(abs(ftrue)) > 0.0
    return maximum(abs(fapp - ftrue))/maximum(abs(ftrue))
end

function errorL2AppTrue(fapp, ftrue)
    @assert length(fapp) == length(ftrue)
    errs = zeros(Float64, (length(fapp),))
    norms = zeros(Float64, (length(fapp),))
    for i = 1:length(fapp)
        errs[i] = sum((ftrue[i] - fapp[i]).^2)
        norms[i] = sum(ftrue[i].^2)
    end
    return sqrt(sum(errs)/sum(norms))
end

function errorLInfAppTrue(fapp, ftrue)
    @assert length(fapp) == length(ftrue)
    errs = zeros(Float64, (length(fapp),))
    for i = 1:length(fapp)
        errs[i] = maximum(abs(ftrue[i] - fapp[i]))/maximum(abs(ftrue[i]))
    end
    return maximum(errs)
end

# In : Array{Array{T, 1}, 1}(dims)
# Out : Array{T, 2}(dims, npts)
function tensor_grid{T}(d1_grids::Array{Array{T,1},1})
    dims = length(d1_grids)
    lengths = zeros(Int64, dims)
    for d = 1:dims
        lengths[d] = length(d1_grids[d])
    end
    tensor_grids = Array{T,2}(dims, prod(lengths))
    points = product(d1_grids...)
    for (i, pts) in enumerate(points)
        tensor_grids[:,i] = [pts...]
    end
    return tensor_grids
end

function cur(A, tol)
    return cur2(A, A, tol)
end

function cur2(Ax, Ay, tol)
    (QFx, RFx, pFx) = qr(Ax', Val{true})
    nPx = cut_spectrum_l2(abs.(diag(RFx)), tol, bypasssort=true)
    (QFy, RFy, pFy) = qr(Ay, Val{true})
    nPy = cut_spectrum_l2(abs.(diag(RFy)), tol, bypasssort=true)
    nP = max(nPx, nPy)
    px = pFx[1:nP]
    py = pFy[1:nP]
    return (px, py)
end

function err_si_l2(kernel, X, Y, Xhat, Yhat, Ktrue=nothing)
    Kapp  = SI.meshKernelFull(kernel, X, Yhat) * (SI.meshKernelFull(kernel, Xhat, Yhat) \ SI.meshKernelFull(kernel, Xhat, Y))
    if Ktrue == nothing
        Ktrue = SI.meshKernelFull(kernel, X, Y)
    end
    err = norm(Kapp - Ktrue) / norm(Ktrue)
end

function err_si_frob(kernel, X, Y, Xhat, Yhat, Ktrue=nothing)
    Kapp  = SI.meshKernelFull(kernel, X, Yhat) * (SI.meshKernelFull(kernel, Xhat, Yhat) \ SI.meshKernelFull(kernel, Xhat, Y))
    if Ktrue == nothing
        Ktrue = SI.meshKernelFull(kernel, X, Y)
    end
    err = vecnorm(Kapp - Ktrue) / vecnorm(Ktrue)
end

function get_mdv(X, n)
    # Pick an initial vertex, we just pick the first
    N = size(X,2)
    if n > N
        return 1:N
    end
    @assert n >= 1
    p = [1]
    d = zeros(N)
    for i = 1:N
        d[i] = distance_2(X[:,i],X[:,p[1]])[1]
    end
    # The first one is the furthest
    (v,i) = findmax(d)
    p = [i]
    for i = 1:N
        d[i] = distance_2(X[:,i],X[:,p[1]])[1]
    end
    while length(p) < n
        # Add a point
        (v,i) = findmax(d)
        append!(p,i)
        # Update distances
        for j = 1:N
            dist = distance_2(X[:,i],X[:,j])[1]
            d[j] = min(d[j],dist)
        end
    end
    return p
end

function cluster_distance(X, Y)
    
    m = size(X, 2)
    n = size(Y, 2)

    # X diameter
    Xdiam = 0
    for i = 1:m
        for j = i+1:m
            dist = SI.distance_2(X[:,i],X[:,j])
            @assert size(dist) == (1,)
            Xdiam = max(Xdiam, dist[1])
        end
    end
    
    # Y diameter
    Ydiam = 0
    for i = 1:n
        for j = i+1:n
            dist = SI.distance_2(Y[:,i],Y[:,j])
            @assert size(dist) == (1,)
            Ydiam = max(Ydiam, dist[1])
        end
    end

    # X-Y distance
    XYdist = Inf
    for i = 1:m
        for j = 1:n
            XYdist = min(XYdist, SI.distance_2(X[:,i], Y[:,j])[1])
        end
    end

    return XYdist / min(Xdiam, Ydiam)

end

function storage_lr(m, n, r)
    return m*r + n*r + r*r
end

