include("../../src/SI.jl")
using Base.Test
srand(1)

# BoundingBox Unit Testing

# CUBE BOUNDING BOX

# 1) 1d examples
for i = 1:100
    X = randn(1,100)
    bb = SI.CubeBoundingBox{Float64}(X)
    @test bb.dim_x == 1
    @test bb.dim_r == 1
    case1 = norm(bb.origin - Array([minimum(X)])) < 1e-14 && norm(bb.lengths - Array([maximum(X)-minimum(X)])) < 1e-14 && norm(bb.basis - Array([1.]))  < 1e-14
    case2 = norm(bb.origin - Array([maximum(X)])) < 1e-14 && norm(bb.lengths - Array([maximum(X)-minimum(X)])) < 1e-14 && norm(bb.basis - Array([-1.])) < 1e-14
    @test case1 || case2
end

# 2) Points on a 2d cube
X = Array{Float64}([0. 0. 1. 1. ; 0. 1. 0. 1.])
bb = SI.CubeBoundingBox{Float64}(X)
@test bb.dim_x == 2
@test bb.dim_r == 2
@test bb.origin == Array([0.0, 0.0])
@test bb.ends == Array([1.0, 1.0])
@test bb.lengths == Array([1.0, 1.0])
@test (bb.basis == Array([1.0 0.0;0.0 1.0]) || bb.basis == Array([0.0 1.0;1.0 0.0]))
@test SI.RtoX(bb,Array([0.5 ; 0.5])) == Array([0.5 ; 0.5])
@test SI.RtoX(bb,Array([0.0 ; 0.0])) == Array([0.0 ; 0.0])

X = Array{Float64}([0. 0. 0. 0. 1. 1. 1. 1. ; 0. 0. 1. 1. 0. 0. 1. 1. ; 0. 1. 0. 1. 0. 1. 0. 1.])
bb = SI.CubeBoundingBox{Float64}(X)
@test bb.dim_x == 3
@test bb.dim_r == 3
@test bb.origin == Array([0., 1., 0.])
@test bb.lengths == Array([1.0, 1.0, 1.0])
@test norm(bb.basis - Array([0. 1. 0. ; 0. 0. -1 ; 1. 0. 0.])) < 1e-14

# 3) Check XtoR and RtoX
for i = 1:200
    d = rand(1:10)
    X = rand(d, 20)
    bb = SI.CubeBoundingBox{Float64}(X)
    Rx = SI.XtoR(bb, X)
    # 3a) Check inside
    for di = 1:d
        @test all(Rx[di,:] .<= bb.ends[di] + 1e-14)
        @test all(Rx[di,:] .>= bb.starts[di] - 1e-14) 
    end
    # 3b) Check RtoX(XtoR(x)) == x
    XX = randn(d, 100)
    RX = SI.XtoR(bb, XX)
    XRX = SI.RtoX(bb, RX)
    @test norm(XRX - XX)/norm(XX) < 1e-14
end

# 4) Check recompression
for i = 1:100
    d = rand(2:10)
    dred = rand(1:d-1)
    Xlow = rand(dred, 20)
    basis = rand(d, dred)
    X = basis * Xlow 
    bb = SI.CubeBoundingBox{Float64}(X, compress=true)
    @test bb.dim_x == d
    @test bb.dim_r == dred
    @test length(bb.ends) == dred
    @test length(bb.lengths) == dred
    @test length(bb.starts) == dred
    Rx = SI.XtoR(bb, X)
    @test size(Rx) == (dred, 20)
    # 3a) Check inside
    for di = 1:dred
        @test all(Rx[di,:] .<= bb.ends[di] + 1e-12)
        @test all(Rx[di,:] .>= bb.starts[di] - 1e-12) 
    end
    # 3b) Check RtoX(XtoR(x)) == x
    XX = basis * randn(dred, 100) # Points have to lie on the plane
    RX = SI.XtoR(bb, XX)
    @test size(RX) == (dred, 100)
    XRX = SI.RtoX(bb, RX)
    @test size(XRX) == (d, 100)
    @test norm(XRX - XX)/norm(XX) < 1e-14
end

# ELLIPSOID BOUNDING BOX

# 1) sub-8th of a sphere
for i = 1:100

    ra = pi/4 + rand() * pi/4
    rb = pi/4 + rand() * pi/4
    l1 = 2.0+rand()
    l2 = 1.0+rand()
    l3 = rand()
    
    center = randn(3)
    a = rand(100) * ra 
    b = rand(100) * rb
    X = vcat( center[1] + l1 * (cos.(a).*sin.(b))' , center[2] + l2 * (sin.(a).*sin.(b))' , center[3] + l3 * (cos.(b))' )
    bb = SI.EllipsoidBoundingBox{Float64}(X)
    bnda = (minimum(a), maximum(a))
    bndb = (minimum(b), maximum(b))

    # 1a) Check invariants

    # Check ranges
    # a range should be < ra
    @test bb.ends[1] >= bb.starts[1]
    @test bb.ends[1] - bb.starts[1] <= ra
    # b range should be < rb
    @test bb.ends[2] >= bb.starts[2]
    @test bb.ends[2] - bb.starts[2] <= rb
    # r range should be ~ 0
    @test bb.ends[3] >= bb.starts[3]
    @test bb.ends[3] - bb.starts[3] <= 4*1e-8
    # Check center : should be ~ 0
    @test norm(bb.center - center) <= 1e-8
    # Check orientation : should eye, +- sign
    @test norm(abs.(bb.basis) - eye(3)) <= 1e-8
    # Check orientation : should be a positive basis
    @test abs.(det(bb.basis)-1) <= 1e-8
    
    # 1b) Check mapping
    
    # XtoR : 
    atrial = bnda[1] + rand(100) * (bnda[2]-bnda[1]) 
    btrial = bndb[1] + rand(100) * (bndb[2]-bndb[1])
    X = vcat( center[1] + l1 * (cos.(atrial).*sin.(btrial))' , center[2] + l2 * (sin.(atrial).*sin.(btrial))' , center[3] + l3 * cos.(btrial)' )
    R = SI.XtoR(bb, X)
    # Check R corresponds to atrial and btrial
    # may be flipped in a
    errA = min( maximum(abs.(R[1,:] - atrial)) , maximum(abs.(R[1,:]+pi-atrial)) )
    @test errA < 1e-6
    @test norm(R[2,:] - btrial) < 1e-6
    @test norm(R[3,:] - ones(100)) < 1e-6

    # RtoX :
    X2 = SI.RtoX(bb, R)
    @test norm(X2 - X)/norm(X) < 1e-6

end
